/**
  ******************************************************************************
  * @file    flashloader.c
  * @author  Milandr Application Team
  * @version V1.1.0
  * @date    09/12/2024
  * @brief   Flashloader for 1986VE4 microcontrollers.
  ******************************************************************************
  * <br><br>
  * THE PRESENT FIRMWARE IS FOR GUIDANCE ONLY. IT AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING MILANDR'S PRODUCTS IN ORDER TO FACILITATE
  * THE USE AND SAVE TIME. MILANDR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES RESULTING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR A USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2024 Milandr</center></h2>
  */

/* Includes ------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "flash_loader.h"
#include "flash_loader_extra.h"

#include "MDR32F9Qx_config.h"
#include "MDR32F9Qx_eeprom.h"
#include "MDR32F9Qx_rst_clk.h"
#include "MDR32F9Qx_utils.h"
#include "MDR32F9Qx_bkp.h"
#include "bootloader.h"

#define PAGE_SIZE               0x200
#define BLOCK_SIZE              0x8000
#define PAGES_TO_BOOT           3
#define CNT_PAGE                4
#define CNT_BLOCK               4
#define ADDR_PAGE(n)            (PAGE_SIZE*n)
#define ADDR_BLOCK(n)           (BLOCK_SIZE*n)

/* Private function prototypes -----------------------------------------------*/
void EEPROM_ReadArray(uint32_t addr, uint32_t *buff, uint32_t count, EEPROM_Mem_Bank BankSelector);
void EEPROM_EraseAllPages(EEPROM_Mem_Bank BankSelector);
void EEPROM_EraseInfoBlock(uint32_t addr_block);
void BKP_SetDefaultTrimmings(void);

ErrorStatus CheckBootloader(void);
void        RestoreBootloader(void);

/* Private variables ---------------------------------------------------------*/
static __no_init char err_buff[100];
static __no_init uint32_t mass_erase_done;

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes the flash driver.
  * @param  base_of_flash: Points to the first byte of the whole flash memory range.
  * @param  image_size: Specifies the size of the whole image that is to be written to flash memory, in bytes.
  * @param  link_address: Specifies the original link address of the first byte of the image, before any offsets 
  * @param  flags: Specifies optional flags.
  * @param  argc: Number of arguments. Only used if USE_ARGC_ARGV is set.
  * @param  argv: An array of argument string pointers. Only used if USE_ARGC_ARGV is set.
  * @retval RESULT_OK - init successful; otherwise returns an error code.
  */
#if USE_ARGC_ARGV
uint32_t FlashInit(void *base_of_flash, uint32_t image_size, uint32_t link_address, uint32_t flags, int argc, char const *argv[])
{
    uint8_t restore_boot = 0;
#else
uint32_t FlashInit(void *base_of_flash, uint32_t image_size, uint32_t link_address, uint32_t flags)
{
#endif
    uint32_t result = RESULT_OK;
    mass_erase_done = 0;

    __disable_irq();

    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, ENABLE);
    RST_CLK_LSIcmd(DISABLE); /* To stop IWDG */
    BKP_SetDefaultTrimmings();

    RST_CLK_DeInit();
    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, ENABLE);

#if defined (_USE_DEBUG_UART_)
    if(STDIO_Init() == ERROR)
    {
        sprintf(err_buff, "Debug UART initialization failed!");
        strcpy(ERROR_MESSAGE_BUFFER, err_buff);
        return RESULT_ERROR_WITH_MSG;
    }
    printf("FlashInit: base_of_flash = 0x%X, image_size = 0x%X, link_address = 0x%X, flags = 0x%X\n\r", (uint32_t)base_of_flash, image_size, link_address, flags);
#endif /* _USE_DEBUG_UART_ */

#if USE_ARGC_ARGV
    for(int i = 0; i < argc; i++)
    {
        if(strcmp("--restore_boot", argv[i]) == 0)
        {
            restore_boot = 1;
        }
    }
#if defined (_USE_DEBUG_UART_)
    printf("FlashInit, Args: restore_boot = %d\n\r", restore_boot);
#endif /* _USE_DEBUG_UART_ */
#endif

    if(flags & (FLAG_ERASE_ONLY | FLAG_MASS_ERASE))
    {
        if(flags & FLAG_MASS_ERASE)
        {
            EEPROM_EraseAllPages(EEPROM_Main_Bank_Select);
            mass_erase_done = 1;
        }
        else
        {
#if USE_ARGC_ARGV
            result = RESULT_ERASE_DONE;
            if(restore_boot)
            {
                if(CheckBootloader() == ERROR)
                {
                    RestoreBootloader();
                    if(CheckBootloader() == ERROR)
                    {
                        result = RESULT_ERROR_WITH_MSG;
                    }
                }
            }
            else
            {
                EEPROM_EraseAllPages(EEPROM_Main_Bank_Select);
            }
#else
            EEPROM_EraseAllPages(EEPROM_Main_Bank_Select);
            result = RESULT_ERASE_DONE;
#endif
        }
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashInit done! Result = 0x%X\n\r", result);
#endif /* _USE_DEBUG_UART_ */

    return result;
}

/**
  * @brief  Writes a data buffer in the internal flash.
  * @param  block_start: Points to the first byte of the block into which this write operation writes.
  * @param  offset_into_block: Specifies how far into the current block that this write operation shall start.
  * @param  count: Specifies the number of bytes to write.
  * @param  buffer: A pointer to the buffer that contains the bytes to write.
  * @retval RESULT_OK - write successful; RESULT_ERROR - write fail.
  */
uint32_t FlashWrite(void *block_start, uint32_t offset_into_block, uint32_t count, char const *buffer)
{
    uint32_t size = 0;

#if defined (_USE_DEBUG_UART_)
    printf("FlashWrite: block_start = 0x%X, offset_into_block = 0x%X, count = 0x%X, buffer = 0x%X\n\r", (uint32_t)block_start, offset_into_block, count, (uint32_t)buffer);
#endif /* _USE_DEBUG_UART_ */

    while(size < count)
    {
        EEPROM_ProgramWord( (uint32_t)block_start + offset_into_block + size, EEPROM_Main_Bank_Select, *((uint32_t *)buffer));
        buffer += 4;
        size += 4;
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashWrite done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Erase block in the internal flash.
  * @param  block_start: Points to the first byte of the block to erase.
  * @param  block_size: Specifies the size of the block, in bytes.
  * @retval RESULT_OK - erase successful; RESULT_ERROR - erase fail.
  */
uint32_t FlashErase(void *block_start, uint32_t block_size)
{
#if defined (_USE_DEBUG_UART_)
    printf("FlashErase: block_start = 0x%X, block_size = 0x%X, mass_erase_done = 0x%X\n\r", (uint32_t)block_start, block_size, mass_erase_done);
#endif /* _USE_DEBUG_UART_ */

    if(mass_erase_done == 0)
    {
        EEPROM_ErasePage((uint32_t)block_start, EEPROM_Main_Bank_Select);
    }

#if defined (_USE_DEBUG_UART_)
    printf("FlashErase done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Calculates the checksum of the downloaded contents of flash memory.
  * @param  begin: Points to the first byte of the block to calculate the checksum.
  * @param  count: Specifies the size of the block, in bytes.
  * @retval calculated checksum value.
  */
OPTIONAL_CHECKSUM
uint32_t FlashChecksum(void const *begin, uint32_t count)
{
    uint16_t crc16 = 0;

#if defined (_USE_DEBUG_UART_)
    printf("FlashChecksum: begin = 0x%X, count = 0x%X\n\r", (uint32_t)begin, count);
#endif /* _USE_DEBUG_UART_ */

    EEPROM_UpdateDCache();
    crc16 = Crc16((uint8_t const *)begin, count);

#if defined (_USE_DEBUG_UART_)
    printf("FlashChecksum done! CRC16 = 0x%X\n\r", crc16);
#endif /* _USE_DEBUG_UART_ */
    
    return crc16;
}

/**
  * @brief  Clearing after the flash is loaded. The function is called after the last call to FlashWrite 
  *         or after FlashChecksum if it exists.
  * @param  None.
  * @retval RESULT_OK.
  */
OPTIONAL_SIGNOFF
uint32_t FlashSignoff()
{
#if defined (_USE_DEBUG_UART_)
    printf("FlashSignoff\n\r");
#endif /* _USE_DEBUG_UART_ */

    EEPROM_UpdateDCache();
    RST_CLK_PCLKcmd(RST_CLK_PCLK_EEPROM, DISABLE);

    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, ENABLE);
    RST_CLK_LSIcmd(ENABLE);
    RST_CLK_PCLKcmd(RST_CLK_PCLK_BKP, DISABLE);

    __enable_irq();

#if defined (_USE_DEBUG_UART_)
    printf("FlashSignoff done! Result = 0x%X\n\r", RESULT_OK);
#endif /* _USE_DEBUG_UART_ */

    return RESULT_OK;
}

/**
  * @brief  Set LDO, HSI and LSI trimmings to their default reset values.
  * @param  None.
  * @retval None.
  */
void BKP_SetDefaultTrimmings(void)
{
    BKP_SetTrim(BKP_TRIM_1_8_V);
    BKP_DUccTrim(BKP_DUcc_plus_100mV);
    RST_CLK_HSIadjust(0x20);
    RST_CLK_LSIadjust(0x10);
}

/**
  * @brief  Check the bootloader in the information area of Flash memory.
  * @param  None.
  * @retval @ref ErrorStatus - SUCCESS if the bootloader is present, else ERROR.
  */
ErrorStatus CheckBootloader(void)
{
    uint32_t index = 0, data = 0;

    for(index = 0; index < sizeof(bootloader) / 4; index++)
    {
        data = EEPROM_ReadWord(index * 4, EEPROM_Info_Bank_Select);
        if (data != bootloader[index])
        {
            sprintf(err_buff, "Bootloader verify error at 0x%X: Flash = 0x%X, Required = 0x%X", index * 4, data, bootloader[index]);
            strcpy(ERROR_MESSAGE_BUFFER, err_buff);
            return (ERROR);
        }
    }

    return (SUCCESS);
}

/**
  * @brief  Restore the bootloader in the information area of Flash memory.
  * @param  None.
  * @retval None.
  */
void RestoreBootloader(void)
{
    uint32_t index = 0;

    for(index = 0; index < PAGE_SIZE * PAGES_TO_BOOT; index += PAGE_SIZE)
    {
        EEPROM_ErasePage(index, EEPROM_Info_Bank_Select);
    }
    for(index = 0; index < sizeof(bootloader) / 4; index++)
    {
        EEPROM_ProgramWord(index * 4, EEPROM_Info_Bank_Select, bootloader[index]);
    }
}

/**
  * @brief  Reads data from EEPROM memory into the data buffer.
  * @param  addr: The EEPROM memory word address (four byte aligned).
  * @param  buffer: Pointer to the buffer to read.
  * @param  count: Specifies the number of bytes to read.
  * @param  BankSelector - @ref EEPROM_Mem_Bank - Selects EEPROM Bank (Main Bank or Information Bank).
  *         This parameter can be one of the following values:
  *             @arg EEPROM_Main_Bank_Select:      The EEPROM Main Bank selector.
  *             @arg EEPROM_Info_Bank_Select:      The EEPROM Information Bank selector.
  * @retval None.
  */
void EEPROM_ReadArray(uint32_t addr, uint32_t *buff, uint32_t count, EEPROM_Mem_Bank BankSelector)
{
    uint32_t size = 0;

    while(size < count)
    {
        *buff = EEPROM_ReadWord(addr + size, BankSelector);
        buff += 1;
        size += 4;
    }
}

/**
  * @brief  Erases all pages of the selected EEPROM memory bank.
  * @param  BankSelector - @ref EEPROM_Mem_Bank - Selects EEPROM Bank (Main Bank, Information Bank or Main and Information Banks).
  *         This parameter can be one of the following values:
  *             @arg EEPROM_Main_Bank_Select:      The EEPROM Main Bank selector.
                @arg EEPROM_Info_Bank_Select:      The EEPROM Information Bank selector.
  *             @arg EEPROM_All_Banks_Select:      The EEPROM Main and Information Banks selector.
  * @retval None
  */
void EEPROM_EraseAllPages(EEPROM_Mem_Bank BankSelector)
{
    uint32_t addr_block = 0, addr_page = 0;

    if((BankSelector == EEPROM_All_Banks_Select) || (BankSelector == EEPROM_Main_Bank_Select))
    {
        for(addr_block = ADDR_BLOCK(0); addr_block < ADDR_BLOCK(CNT_BLOCK); addr_block += BLOCK_SIZE)
        {
            EEPROM_EraseBlock(addr_block, BankSelector);
        }
    }
    else if(BankSelector == EEPROM_Info_Bank_Select)
    {
        for (addr_block = ADDR_BLOCK(0); addr_block < ADDR_BLOCK(CNT_BLOCK); addr_block += BLOCK_SIZE)
        {
            for(addr_page = ADDR_PAGE(0); addr_page < ADDR_PAGE(CNT_PAGE); addr_page += PAGE_SIZE)
            {
                EEPROM_ErasePage(addr_block + addr_page, EEPROM_Info_Bank_Select);
            }
        }
    }
}

/**
  * @brief  Erases one block of the EEPROM Information Bank.
  * @param  Address: Block Address in the EEPROM memory.
  * @retval None
  */
void EEPROM_EraseInfoBlock(uint32_t addr_block)
{
    uint32_t addr_page = 0;

    for(addr_page = ADDR_PAGE(0); addr_page < ADDR_PAGE(CNT_PAGE); addr_page += PAGE_SIZE)
    {
        EEPROM_ErasePage(addr_block + addr_page, EEPROM_Info_Bank_Select);
    }
}

